// 3D Modeling Courses Website - Version 4 (HTML/CSS/JS) - Minimalist Clean Design

const programs = {
  blender: {
    name: 'Blender',
    color: '#1A1F2E',
    instructor: 'Алексей Волков'
  },
  maya: {
    name: 'Maya',
    color: '#1A1F2E',
    instructor: 'Мария Соколова'
  },
  '3dsmax': {
    name: '3ds Max',
    color: '#1A1F2E',
    instructor: 'Дмитрий Новиков'
  },
  zbrush: {
    name: 'ZBrush',
    color: '#1A1F2E',
    instructor: 'Елена Петрова'
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval
let selectedProgram = ''

function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadProjects()
  updateStats()
  
  setInterval(() => {
    loadProjects()
    updateStats()
  }, 500)
}

function loadProjects() {
  const saved = localStorage.getItem('model3dProjects')
  let projects = []
  
  if (saved) {
    try {
      projects = JSON.parse(saved)
    } catch (e) {
      projects = []
    }
  } else {
    projects = [
      {
        id: 1,
        course: 'Blender для начинающих',
        instructor: 'Алексей Волков',
        submissionDate: '2025-03-28',
        rating: 5,
        progress: 85
      },
      {
        id: 2,
        course: 'Maya Advanced',
        instructor: 'Мария Соколова',
        submissionDate: '2025-03-25',
        rating: 5,
        progress: 92
      },
      {
        id: 3,
        course: '3ds Max Профессионал',
        instructor: 'Дмитрий Новиков',
        submissionDate: '2025-03-22',
        rating: 4,
        progress: 78
      }
    ]
    localStorage.setItem('model3dProjects', JSON.stringify(projects))
  }
  
  const statsClean = document.getElementById('statsClean')
  if (statsClean) {
    const totalProjects = projects.length
    const avgRating = projects.length > 0 ? (projects.reduce((sum, p) => sum + p.rating, 0) / projects.length).toFixed(1) : 0
    const totalProgress = projects.length > 0 ? Math.round(projects.reduce((sum, p) => sum + p.progress, 0) / projects.length) : 0
    const completedProjects = projects.filter(p => p.progress === 100).length
    
    statsClean.innerHTML = `
      <div class="stat-clean-item">
        <div class="stat-clean-icon">
          <img src="icons/icon-box.svg" alt="Box" width="32" height="32">
        </div>
        <div class="stat-clean-content">
          <div class="stat-clean-value">${totalProjects}</div>
          <div class="stat-clean-label">Проектов</div>
        </div>
      </div>
      <div class="stat-clean-item">
        <div class="stat-clean-icon">
          <img src="icons/icon-star-fill.svg" alt="Star" width="32" height="32">
        </div>
        <div class="stat-clean-content">
          <div class="stat-clean-value">${avgRating}</div>
          <div class="stat-clean-label">Рейтинг</div>
        </div>
      </div>
      <div class="stat-clean-item">
        <div class="stat-clean-icon">
          <img src="icons/icon-trending-up.svg" alt="Trending" width="32" height="32">
        </div>
        <div class="stat-clean-content">
          <div class="stat-clean-value">${totalProgress}%</div>
          <div class="stat-clean-label">Прогресс</div>
        </div>
      </div>
      <div class="stat-clean-item">
        <div class="stat-clean-icon">
          <img src="icons/icon-award.svg" alt="Award" width="32" height="32">
        </div>
        <div class="stat-clean-content">
          <div class="stat-clean-value">${completedProjects}</div>
          <div class="stat-clean-label">Завершено</div>
        </div>
      </div>
    `
  }
  
  const projectsList = document.getElementById('projectsList')
  if (projectsList) {
    if (projects.length === 0) {
      projectsList.innerHTML = `
        <div class="empty-clean">
          <div class="empty-clean-icon">
            <img src="icons/icon-box.svg" alt="Box" width="56" height="56">
          </div>
          <h3 class="empty-clean-title">Начните свой путь в 3D</h3>
          <p class="empty-clean-text">Запишитесь на курс и увидите свой первый проект здесь</p>
        </div>
      `
    } else {
      projectsList.innerHTML = projects.map((project, index) => {
        const stars = Array.from({ length: 5 }, (_, i) => 
          `<img src="icons/icon-star${i < project.rating ? '-fill' : ''}.svg" alt="Star" width="16" height="16">`
        ).join('')
        
        const date = new Date(project.submissionDate).toLocaleDateString('ru-RU', { 
          day: 'numeric', 
          month: 'long', 
          year: 'numeric' 
        })
        
        return `
          <div class="project-clean-item">
            <div class="project-clean-indicator"></div>
            <div class="project-clean-content">
              <div class="project-clean-header">
                <div class="project-clean-badge">${project.course.split(' ')[0]}</div>
                <div class="project-clean-rating">${stars}</div>
              </div>
              <h3 class="project-clean-course">${project.course}</h3>
              <div class="project-clean-details">
                <div class="project-clean-detail">
                  <img src="icons/icon-users.svg" alt="User" width="16" height="16">
                  <span>${project.instructor}</span>
                </div>
                <div class="project-clean-detail">
                  <img src="icons/icon-calendar.svg" alt="Calendar" width="16" height="16">
                  <span>${date}</span>
                </div>
              </div>
              <div class="project-clean-progress">
                <div class="project-clean-progress-header">
                  <span class="progress-clean-label">Прогресс</span>
                  <span class="progress-clean-value">${project.progress}%</span>
                </div>
                <div class="project-clean-progress-bar">
                  <div class="project-clean-progress-fill" style="width: ${project.progress}%;"></div>
                </div>
              </div>
            </div>
          </div>
        `
      }).join('')
    }
  }
}

function updateStats() {
  // Stats are updated in loadProjects
}

function initEnrollPage() {
  const programCards = document.querySelectorAll('.program-clean-card')
  programCards.forEach(card => {
    card.addEventListener('click', () => {
      const program = card.dataset.program
      selectProgram(program)
    })
  })
  
  const form = document.getElementById('enrollForm')
  if (form) {
    form.addEventListener('submit', handleSubmit)
  }
  
  const startDateInput = document.getElementById('startDate')
  if (startDateInput) {
    const today = new Date()
    const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
    startDateInput.min = today.toISOString().split('T')[0]
    startDateInput.max = maxDate.toISOString().split('T')[0]
    startDateInput.addEventListener('change', updateSubmitButton)
  }
}

function selectProgram(programId) {
  selectedProgram = programId
  
  const programCards = document.querySelectorAll('.program-clean-card')
  programCards.forEach(card => {
    if (card.dataset.program === programId) {
      card.classList.add('selected')
      const program = programs[programId]
      const icon = card.querySelector('.program-clean-icon')
      if (icon) {
        icon.style.background = 'rgba(26, 31, 46, 0.1)'
        icon.style.borderColor = program.color
      }
    } else {
      card.classList.remove('selected')
      const program = programs[programId]
      const icon = card.querySelector('.program-clean-icon')
      if (icon) {
        icon.style.background = 'rgba(26, 31, 46, 0.05)'
        icon.style.borderColor = 'rgba(26, 31, 46, 0.2)'
      }
    }
  })
  
  const dateSection = document.getElementById('dateSection')
  if (dateSection) {
    dateSection.style.display = 'block'
  }
  
  updateSubmitButton()
}

function updateSubmitButton() {
  const submitBtn = document.getElementById('submitBtn')
  const startDate = document.getElementById('startDate')
  if (submitBtn && startDate) {
    if (selectedProgram && startDate.value) {
      submitBtn.disabled = false
    } else {
      submitBtn.disabled = true
    }
  }
}

function handleSubmit(e) {
  e.preventDefault()
  
  const startDate = document.getElementById('startDate')
  if (!startDate || !startDate.value) {
    return
  }
  
  const saved = localStorage.getItem('model3dProjects')
  const projects = saved ? JSON.parse(saved) : []
  
  const program = programs[selectedProgram]
  const instructor = program ? program.instructor : 'Преподаватель'
  
  const newProject = {
    id: Date.now(),
    course: program ? program.name : 'Курс',
    instructor: instructor,
    submissionDate: startDate.value,
    rating: 0,
    progress: 0
  }
  
  projects.push(newProject)
  localStorage.setItem('model3dProjects', JSON.stringify(projects))
  
  const successMessage = document.getElementById('successMessage')
  if (successMessage) {
    successMessage.style.display = 'flex'
    setTimeout(() => {
      window.location.href = 'index.html'
    }, 2000)
  }
}

function initSlider() {
  slides = Array.from(document.querySelectorAll('.slide'))
  dots = Array.from(document.querySelectorAll('.dot'))
  
  if (slides.length === 0) return
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')
  
  if (sliderPrev) {
    sliderPrev.addEventListener('click', () => {
      goToSlide((currentSlide - 1 + slides.length) % slides.length)
    })
  }
  
  if (sliderNext) {
    sliderNext.addEventListener('click', () => {
      goToSlide((currentSlide + 1) % slides.length)
    })
  }
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      goToSlide(index)
    })
  })
  
  startSlider()
}

function goToSlide(index) {
  currentSlide = index
  
  slides.forEach((slide, i) => {
    slide.classList.toggle('active', i === index)
  })
  
  dots.forEach((dot, i) => {
    dot.classList.toggle('active', i === index)
  })
  
  clearInterval(slideInterval)
  startSlider()
}

function startSlider() {
  slideInterval = setInterval(() => {
    goToSlide((currentSlide + 1) % slides.length)
  }, 3000)
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

